/* $Id: install.c,v 1.76 1998/12/16 20:48:09 ericb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Chris Sutton and Eric Backus */

#include "sema.h"


#if !defined(E1485_SOURCE) && !defined(HPVXI_DOWNLOAD)
#include <fcntl.h>
#endif

#ifdef	MICROSOFT_OS
#  include <io.h>
#  ifdef __STDC__       /* non-ansi names used in this file */
#    define open _open
#    define close _close
#    define lseek _lseek
#    define read _read
#    define O_RDONLY _O_RDONLY
#    define O_BINARY _O_BINARY
#  endif
#else
#  ifdef	E1485_SOURCE
#    define O_RDONLY	0
#  else
#    ifndef HPVXI_DOWNLOAD
#      include <unistd.h>
#    endif
#  endif
#  define O_BINARY	0
#endif

#ifndef HPVXI_DOWNLOAD
static char errInfo[80];
#endif	/* HPVXI_DOWNLOAD */

#define	BUFFER_SIZE			8192
#define E1432_HOSTB_ICS_TEST_MAX        1000

#define CHK(func) error = (func); if (error) return error;

/*
 *********************************************************************
 This function is used to open a file, and determine its size.  The
 file descriptor is returned in <file_des>, the file size is returned
 in <file_size>.  This function returns zero if successful, or an
 error if not.
 *********************************************************************
 */
SHORTSIZ16
i1432_open_file(const char *path, int *file_des, off_t *file_size)
{
#ifndef HPVXI_DOWNLOAD
    LONGSIZ32 status;

    TRACE_PRINTF(2, ("    i1432_open_file(%s)\n", path));
    DIAG_PRINTF(2, ("i1432_open_file(%s)\n", path));

    /* Open executable file */
    *file_des = open(path, O_RDONLY | O_BINARY);

    if (*file_des == -1)
    {
        (void) sprintf(errInfo, "failed opening file %s", path);
	i1432_error_info = errInfo;
	return i1432_print_error(ERR1432_FILE);
    }

    /* Get file size */
    *file_size = lseek(*file_des, 0L, 2);
    if (*file_size <= 0)
	return i1432_print_error(ERR1432_FILE);

    status = lseek(*file_des, 0L, 0);
    if (status < 0)
	return i1432_print_error(ERR1432_FILE);

#endif /* HPVXI_DOWNLOAD */
    return 0;
}

/*
 *********************************************************************
 Installs code into an E1432 module.  If <file_des> is -1, then <buf1>
 points to a memory array containing the data to download.  Otherwise,
 <file_des> is the file descriptor of a file containing the data to
 download.  <file_size> specifies how many bytes will be downloaded.
 <buf2> must point to an already allocated memory buffer that will be
 used during the download process.

 e1432_init_io_driver() should be called before this function.
 e1432_assign_channel_numbers() should be called after.

 Currently, this routine handshakes the first block by checking the
 96000 TXDE bit.  It may need to handshake each DSP word to avoid
 sending code faster than the 96000 can take it.  After that,
 *********************************************************************
 */
static SHORTSIZ16
i1432_install(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 flags,
	      int file_des, off_t file_size,
	      char *buf1, char *buf2)
{
    LONGSIZ32 ics;
#ifdef	HAVE_SICL
    LONGSIZ32 status;
#endif
    SHORTSIZ16 error, page, error2;
    size_t  i, j, sub_xfer_size;
    unsigned char *pin;
    unsigned short *pout;
    unsigned long bytes_sent, window;
    unsigned long dsp_mem, dsp_mem_start;
      /*  Was: unsigned short *dsp_mem, *dsp_mem_start;, which causes
       *  problems when Windows 3.1 does its huge pointer segment/offset
       *  tricks.  It is best to handle external (VXI or module) addresses
       *  as longs, not as pointers.
       */
    int *dec_ptr;	/* globals for decompression routines */
    int from_mem = (flags & E1432_INSTALL_FROM_MEM) != 0;
    int decompress_mem = from_mem && (flags & E1432_INSTALL_DECOMPRESS) != 0;
#ifndef	HAVE_VTL
    unsigned short *rxtxAddr;
#endif
#if SWAP
    unsigned char *poutchar;
#endif

    TRACE_PRINTF(2, ("    i1432_install(%d)\n", mn->la));
    DIAG_PRINTF(2, ("i1432_install(%d)\n", mn->la))

    /* Reset the module */
    CHK(i1432_reset(mn));
    CHK(i1432_wait_reset(mn));

    DIAG_PRINTF(3, ("  setting up hostport\n"));
    /* See if the host port is activated */
    CHK(i1432_wait32_bits_match(mn, E1432_HOSTB_ICS_REG,
				E1432_HOSTB_ICS_HRST, 0, 0.01,
				"Host port in reset, change DIP switches?"));

    /* Get 96000 host port ICS register */
    CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG, &ics));


    /* Clear HF1, signalling 8-bit transfers */
    ics &= ~E1432_HOSTB_ICS_HF1;

    CHK(i1432_direct_write32_register(mn, E1432_HOSTB_ICS_REG, ics));

    /* Kick off download if bootrom is running */
    CHK(i1432_wait32_bits_match(mn, E1432_HOSTB_CVR_REG,
				E1432_HOSTB_CVR_HC, 0, 0.01,
				"Host port dead, change DIP switches?"));
    CHK(i1432_direct_write32_register(mn, E1432_HOSTB_CVR_REG,
				      E1432_HOSTB_CVR_HOSTACMD));

    /* load from memory setups */
    if ( from_mem )
    {
        if ( decompress_mem )
	{
            /* initialize decompression from file_des[] */
	    CHK((SHORTSIZ16) i1432_decompress_init((char *)file_des,
						   &dec_ptr,
						   file_size));
        }
	else
	{
	    /* no decompression, point buf1 directly to file_des[] */
	    buf1 = (char *)file_des;
	}
    }

#ifndef	HPVXI_DOWNLOAD
    if ( ! from_mem )
	/* Reset the file pointer */
	if (lseek(file_des, 0L, 0) < 0)
	    return i1432_print_error(ERR1432_FILE);
#endif

    page = E1432_A_SRAM_PAGE;
    if (mn->a24_256k)
    {
	window = E1432_WINDOW_BASE_256K;
	page <<= 2;
    }
    else
	window = E1432_WINDOW_BASE_1M;


#ifdef	HAVE_VTL
    dsp_mem_start = 0;
#else
    if (mn->a24_256k)
	/* When using a24_256k, we map in only the movable window,
	   sothe address we want is just the base address that was
	   imapped. */
	dsp_mem_start = 0;
    else
	/* When not using a24_256k, we map all of the module's A24
	   space, so we want the address of the movable window. */
	dsp_mem_start = window;

    /* Get the 96002 host port address to write data to */
    CHK(i1432_calc_register_address(mn, E1432_HOSTB_RXTX_REG + 2,
				    (volatile SHORTSIZ16 **)
				    &rxtxAddr));
#endif

    dsp_mem = dsp_mem_start;



    /* Send the data */
    DIAG_PRINTF(3, ("  downloading\n"));
    bytes_sent = 0;
    while (file_size > 0)
    {
        DIAG_PRINTF(4, ("  bytes_sent = %ld\n", bytes_sent));
	/* Fill at most half the buffer, so we can unpack the bytes below */
	sub_xfer_size = BUFFER_SIZE / 2;

	/* Fill at most 4096, so that there is a pause between the
	   first 4096 block and the rest */
	if (sub_xfer_size > 4096)
	    sub_xfer_size = 4096;

	/* Fill at most file_size bytes, so we don't overrun the
	   input data */
	if ((off_t) sub_xfer_size > file_size)
	     sub_xfer_size = (size_t) file_size;
#ifndef HPVXI_DOWNLOAD
        if ( ! from_mem )
	{
	    /* Get another block of data */
	    if (read(file_des, buf1, sub_xfer_size)
		!= (int) sub_xfer_size)
		return i1432_print_error(ERR1432_FILE);
	}
#endif
	/* if loading from memory and decompressing */
        if ( decompress_mem )
        {
	    /* decompress another block */
	    (void) i1432_decompress(buf1,
				    (unsigned long) sub_xfer_size,
				    dec_ptr);
	}

#ifndef VISA_PORT_INSTALL /* have VISA load all sema.bin using host port */
	/* Transfer it to the module */
	if (bytes_sent >= 4096)
	{
	    /* Use direct memory access for all but the first 4K
	       bytes, it's safest and fastest. */

#if SWAP
	    /* Scramble the data in just the right way.  The block
	       transfer routine will byte-swap the data, but we don't
	       want our sema.bin image byte-swapped, so we undo it
	       here. */
	    pin = (unsigned char *) buf1;
	    poutchar = (unsigned char *) buf2;
	    for (i = 0; i < sub_xfer_size / 4; i++)
	    {
    		if (mn->d32)
		{
	            *poutchar++ = *(pin + 3);
	            *poutchar++ = *(pin + 2);
	            *poutchar++ = *(pin + 1);
	            *poutchar++ = *(pin + 0);
	            pin += 4;
		}
		else
		{
	            *poutchar++ = *(pin + 1);
	            *poutchar++ = *(pin + 0);
	            *poutchar++ = *(pin + 3);
	            *poutchar++ = *(pin + 2);
	            pin += 4;
		}
	    }
	    pout = (unsigned short *) (void *) buf2;
#else
	    pout = (unsigned short *) (void *) buf1;
#endif

	    DIAG_PRINTF(5, ("  copying block\n"));

	    /* Set page map register to give access to DSP A low */
	    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));


	    error2 = i1432_xfer_block(mn, (unsigned long *) (void *) pout,
				      (unsigned long *) dsp_mem,
				      sub_xfer_size / 4, 0, 0);
	    /* Don't check error2 until after restoring page map below */


	    /* Set page map register to disable access to DSP A low */
	    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0));

	    if (error2 != 0)
	    {
		DIAG_PRINTF(1, ("Download copy failed during transfer.\n"));
		return error2;
	    }
	}
	else
#endif /* ifndef VISA_PORT_INSTALL */
	{
	    /* Can't use direct memory access for the first 4K bytes.
	       Instead, we must go through the 96002 host port. */

	    /* Check for data empty */
	    for (j = 0; j < E1432_HOSTB_ICS_TEST_MAX; j++)
	    {
		CHK(i1432_direct_read32_register(mn,
						 E1432_HOSTB_ICS_REG,
						 &ics));
		if ((ics & E1432_HOSTB_ICS_TXDE) != 0)
		    break;
	    }
	    if (j == E1432_HOSTB_ICS_TEST_MAX)
	    {
#ifndef HPVXI_DOWNLOAD
		(void) sprintf(errInfo, "install failed after sending %ld bytes",
			       bytes_sent);
		i1432_error_info = errInfo;
#endif /* !HPVXI_DOWNLOAD */
                DIAG_PRINTF(1, ("Install failed after %ld bytes.\n",
		  bytes_sent));
		return i1432_la_print_error(mn->la, ERR1432_IO);
	    }

	    /* Unpack the data into one byte per 16-bit word.  The
	       data is big-endian, because it is an image to get
	       loaded into a 96002.  However, the bootstrap program
	       wants LSB first, so we have to swap the order of the
	       bytes. */
	    pin = (unsigned char *) buf1;
	    pout = (unsigned short *) (void *) buf2;
	    for (i = 0; i < sub_xfer_size / 4; i++)
	    {
		*pout++ = *(pin + 3);
		*pout++ = *(pin + 2);
		*pout++ = *(pin + 1);
		*pout++ = *(pin + 0);
		pin += 4;
	    }

            DIAG_PRINTF(5, ("  copying block\n"));
#ifdef	HAVE_SICL
	    if ((flags & E1432_INSTALL_FASTEST) != 0)
	    {
		/* Not safe, but avoids separate pokes and handshake
		   for each poke.  On a slow enough interface, it is
		   worthwhile to do this.  I think only production
		   uses this code, and only on an E1406. */
		status = iwpushfifo(mn->sicl_id,
				    (unsigned short *) (void *) buf2,
				    rxtxAddr, sub_xfer_size, SWAP);
		if (status)
		{
		    (void) sprintf(errInfo,
				   "iwpushfifo failed. Sicl error: %s",
				   igeterrstr((int) status));
		    i1432_error_info = errInfo;
		    return i1432_la_print_error(mn->la, ERR1432_IO);
		}
	    }
	    else
#endif
	    {
		/* Slowest, safest, way */
		volatile unsigned short *p =
		    (volatile unsigned short *) (void *) buf2;
		for (i = 0; i < sub_xfer_size; i++)
		{
		    /* Check for data empty */
		    for (j = 0; j < E1432_HOSTB_ICS_TEST_MAX; j++)
		    {
			CHK(i1432_direct_read32_register(mn,
							 E1432_HOSTB_ICS_REG,
							 &ics));
			if ((ics & E1432_HOSTB_ICS_TXDE) != 0)
			    break;
		    }
		    if (j == E1432_HOSTB_ICS_TEST_MAX)
		    {
#ifndef	HPVXI_DOWNLOAD
			(void) sprintf(errInfo,
				       "install failed after sending %ld bytes",
				       bytes_sent + i);
			i1432_error_info = errInfo;
#endif
			return i1432_la_print_error(mn->la, ERR1432_IO);
		    }
#ifdef	HAVE_VTL
		    CHK(i1432_direct_write_register(mn,
						    E1432_HOSTB_RXTX_REG + 2,
						    (*p++)));
#else
		    iwpoke(rxtxAddr, *p++);
#endif
		}
	    }
            DIAG_PRINTF(3, ("  downloaded first 4k\n"));
#ifdef WIN32
	    /* give monitor time to startup */
	    if(bytes_sent==0)
		Sleep(100);
#endif
	}

    	bytes_sent += sub_xfer_size;
	file_size -= sub_xfer_size;
	/* if loading from memory but not decompressing */
        if ( from_mem && ! decompress_mem )
	    /* Increment buf1 through the memory array */
	    buf1 += sub_xfer_size;

	/* Increment dsp_mem */
	dsp_mem += sub_xfer_size;
	if ((long) dsp_mem - (long) dsp_mem_start >= (long) window)
	{
	    /* We hit the top of the movable window, so move the
	       window and start again at the bottom. */
	    dsp_mem -= window; 
	    page++;
	}
    }
    DIAG_PRINTF(3, ("  finished downloading\n"));

    /* Set HF0 to indicate we're done */
    CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG, &ics));
    ics |= E1432_HOSTB_ICS_HF0;
    CHK(i1432_direct_write32_register(mn, E1432_HOSTB_ICS_REG, ics));

    if ((flags & E1432_INSTALL_SYSCALLS) == 0)
    {
	/* Clear HF1, used as syscall flag by e1432 firmware */
	CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG, &ics));
	ics &= ~E1432_HOSTB_ICS_HF1;
	CHK(i1432_direct_write32_register(mn, E1432_HOSTB_ICS_REG, ics));

	/* Clear HF0, used as syscall flag by e1432 firmware */
	CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG, &ics));
	ics &= ~E1432_HOSTB_ICS_HF0;
	CHK(i1432_direct_write32_register(mn, E1432_HOSTB_ICS_REG, ics));
    }
    else
    {
	/* Set HF0 and HF1, used as syscall flags by e1432 firmware.
	   The firmware startup may have cleared them. */
	CHK(i1432_direct_read32_register(mn, E1432_HOSTB_ICS_REG, &ics));
	ics |= E1432_HOSTB_ICS_HF0 | E1432_HOSTB_ICS_HF1;
	CHK(i1432_direct_write32_register(mn, E1432_HOSTB_ICS_REG, ics));
    }
    DIAG_PRINTF(2, ("i1432_install() completed successfully.\n"));
    return 0;
}

static SHORTSIZ16
i1432_copy_install_file(const char *file)
{
    if ( i1432_install_file != NULL )
    {
	free((void *)i1432_install_file);
    }
    i1432_install_file = (char *)malloc((strlen(file) + 1) * sizeof(char));
    if ( i1432_install_file == NULL )
    {
        return i1432_print_error(ERR1432_MALLOC);
    }
    (void) strcpy((char *)i1432_install_file, file);
    return 0;
}

SHORTSIZ16 EXPORT
e1432_install_file(const char *fileName, SHORTSIZ16 from_mem)
{
    SHORTSIZ16 error;

    if ( from_mem )
    {
	i1432_install_file = fileName;
    }
    else
    {
        CHK(i1432_copy_install_file(fileName));
    }
    i1432_install_from_mem = from_mem;

    return 0;
}

/*
 *********************************************************************
 This function installs code into E1432 modules.
 Reads a file and downloads it to the 96000.

 e1432_init_io_driver() should be called before this function,
 e1432_assign_channel_numbers() should be called after.

 If it fails to find a E1432 module at the logical address, or can't
 install code, it will return a negative error number,
 otherwise it returns 0.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_install(SHORTSIZ16 modCount, SHORTSIZ16 * las, LONGSIZ32 flags,
	      const void *path)
{
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list = NULL;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    SHORTSIZ16 error = 0;
    SHORTSIZ16 error2 = 0;
    off_t   file_size = 0;
    int     file_des = -1;
    char   *buf1 = NULL;
    char   *buf2 = NULL;
    int     i, save_chan_count;
    int from_mem = (flags & E1432_INSTALL_FROM_MEM) != 0;

    TRACE_PRINTF(0, ("e1432_install(%d, 0x%p, 0x%p)\n",
		     modCount, las, path));
    DIAG_PRINTF(2, ("e1432_install(%d, 0x%x, 0x%lx, 0x%x)\n",
      modCount, las, flags, path))

    i1432_dnld_fw_state = 0;  /* substrate FW not downloaded */
    i1432_dnld_fw_err = 0;    /* no substrate FW download error */

    /* Do this before calling i1432_fake_setup_sicl.  Also do it
       before anything which might jump to the cleanup code. */
#if defined(HAVE_SICL) || defined(HAVE_VTL)
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    /* Use previous file name, if none given, then save the file
       name to be re-used. */
    if (path == NULL)
    {
	path = i1432_install_file;
	from_mem = (int)i1432_install_from_mem;
    }
    else
    {
        /* make a copy of the install file */
        if ( from_mem )
        {
	    i1432_install_file = path;
        }
        else
        {
            CHK(i1432_copy_install_file(path));
        }
        i1432_install_from_mem = (SHORTSIZ16) from_mem;
    }

    /* Check for bad modCount parameter */
    if (modCount <= 0 || modCount >= E1432_MODULES)
    {
	i1432_dnld_fw_state = E1432_DNLD_FW_FAILED;
        i1432_dnld_fw_err = ERR1432_ILLEGAL_MODULE_COUNT;
	return i1432_print_error(ERR1432_ILLEGAL_MODULE_COUNT);
    }

    if ( ! from_mem )
    {
	/* Open the code file */
	error = i1432_open_file(path, &file_des, &file_size);
	if (error)
	{
            DIAG_PRINTF(0, ("  Error opening download file \"%s\"\n", path))
	    goto cleanup;
	}
    }
    else
    {
	/* Data comes from passed-in memory buffer */
	file_size = ((struct e1432_install_from_mem *) path)->nbyte;
	file_des = (int)(((struct e1432_install_from_mem *) path)->data);
    }

    /* Allocate buffers for code */
    buf1 = (char *) malloc((size_t) BUFFER_SIZE);
	if (buf1 == NULL)
    error = i1432_print_error(ERR1432_MALLOC);
    if (error)
	goto cleanup;
    buf2 = (char *) malloc((size_t) BUFFER_SIZE);
    if (buf2 == NULL)
	error = i1432_print_error(ERR1432_MALLOC);
    if (error)
	goto cleanup;

    /* Download code to all modules */
    for (i = 0; i < modCount; i++)
    {
	/* Set up SICL for this module */
	/* Set extra_io to zero; module might be completely hosed */
	error = i1432_fake_setup_sicl(mn, las[i], &tmp_chan_list, 0);
	if (error)
	    goto cleanup;
	/* Download the code */
	error = i1432_install(mn, flags, file_des, file_size, buf1, buf2);
	if (error)
	{
	    goto cleanup;
	}
    }
    /* Give module time to clear READY before it sets it again */
    i1432_pause(0.1);

    /* Check all modules for errors */
    for (i = 0; i < modCount; i++)
    {
	/* Set up SICL for this module */
	error = i1432_fake_setup_sicl(mn, las[i], &tmp_chan_list, 1);
	if (error)
	    goto cleanup;

	/* Wait for ready */
	error = i1432_wait_bits_match(mn, E1432_STATUS_REG,
				      E1432_STATUS_READY,
				      E1432_STATUS_READY,
				      11.0, "Waiting for ready");
	if (error)
	    goto cleanup;

	/* Un-inhibit sysfail */
	error = i1432_direct_write_register(mn, E1432_CONTROL_REG,
					(SHORTSIZ16)E1432_CONTROL_A24_ENABLE);
	if (error)
	    goto cleanup;

	/* Check for errors */
	/* Pass in bogus hw and channel ID, which works because of
	   kludges done in i1432_fake_setup_sicl */
	error = i1432_error_check(0, 1);
	if (error)
	    goto cleanup;
    }

cleanup:
    i1432_dnld_fw_err = error;
    if ( error )
    {
        i1432_dnld_fw_state = E1432_DNLD_FW_FAILED;
    }
    else
    {
        i1432_dnld_fw_state = E1432_DNLD_FW_SUCCESSFUL;
    }
    error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    if (error2)
	error = error2;
#ifndef HPVXI_DOWNLOAD
    if ( ! from_mem )
    {
	(void) close(file_des);
    }
#endif  /* HPVXI_DOWNLOAD */
    if (buf1)
        free(buf1);
    if (buf2)
	free(buf2);
    return error;
}
